<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Import Contacts - CSV Upload</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <style>
        .import-container {
            max-width: 800px;
            margin: 2rem auto;
            padding: 2rem;
            background: #fff;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .file-upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 2rem;
            text-align: center;
            transition: all 0.3s ease;
            background: #f8f9fa;
        }
        
        .file-upload-area:hover {
            border-color: #6c04f5;
            background: #f0f0ff;
        }
        
        .file-upload-area.dragover {
            border-color: #6c04f5;
            background: #e6e6ff;
        }
        
        .upload-icon {
            font-size: 3rem;
            color: #6c63ff;
            margin-bottom: 1rem;
        }
        
        .btn-primary {
            background-color: #6c04f5;
            border-color: #6c04f5;
        }
        
        .btn-primary:hover {
            background-color: #5a03d1;
            border-color: #5a03d1;
        }
        
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .progress-container {
            display: none;
            margin-top: 1rem;
        }
        
        #file-name {
            margin-top: 0.5rem;
            font-weight: 500;
            color: #28a745;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container">
        <div class="import-container">
            <!-- Header -->
            <div class="text-center mb-4">
                <h1 class="h3 mb-3">
                    <i class="fas fa-user-plus text-primary"></i>
                    Add Contacts
                </h1>
                <p class="text-muted">Import contacts from CSV or add a single contact manually</p>
            </div>

            <!-- Current Stats -->
            <div class="stats-card">
                <div class="row text-center">
                    <div class="col-md-6">
                        <h4><i class="fas fa-users"></i> {{ number_format($totalContacts) }}</h4>
                        <p class="mb-0">Total Contacts in Database</p>
                    </div>
                    <div class="col-md-6">
                        <h4><i class="fas fa-calendar"></i> {{ date('Y-m-d') }}</h4>
                        <p class="mb-0">Today's Date</p>
                    </div>
                </div>
            </div>

            <!-- Messages -->
            @if (session('success'))
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle"></i>
                    {{ session('success') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            @if (session('error'))
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle"></i>
                    {{ session('error') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            @if ($errors->any())
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle"></i>
                    <ul class="mb-0">
                        @foreach ($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            <!-- Tabs Navigation -->
            <ul class="nav nav-tabs mb-4" id="contactTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="csv-tab" data-bs-toggle="tab" data-bs-target="#csv-upload" type="button" role="tab">
                        <i class="fas fa-upload"></i> CSV Upload
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="single-tab" data-bs-toggle="tab" data-bs-target="#single-contact" type="button" role="tab">
                        <i class="fas fa-user-plus"></i> Single Contact
                    </button>
                </li>
            </ul>

            <!-- Tab Content -->
            <div class="tab-content" id="contactTabsContent">
                <!-- CSV Upload Tab -->
                <div class="tab-pane fade show active" id="csv-upload" role="tabpanel">

                    <!-- Sample Download -->
                    <div class="text-center mb-4">
                        <a href="{{ route('contacts.download-sample') }}" class="btn btn-outline-primary">
                            <i class="fas fa-download"></i> Download Sample CSV
                        </a>
                        <small class="d-block mt-2 text-muted">
                            Download a sample CSV file to see the required format
                        </small>
                    </div>

                    <!-- CSV Format Info -->
                    <div class="card mb-4">
                        <div class="card-body">
                            <h6 class="card-title"><i class="fas fa-info-circle text-info"></i> CSV Format Requirements</h6>
                            <p class="card-text mb-2">Your CSV file should have the following columns in order:</p>
                            <ol class="mb-0">
                                <li><strong>Name</strong> - Full name of the contact</li>
                                <li><strong>Phone No.</strong> - Phone number (with or without country code)</li>
                                <li><strong>Social</strong> - Social media handle (optional)</li>
                                <li><strong>School</strong> - School name (optional, defaults to 'Unknown')</li>
                            </ol>
                        </div>
                    </div>

            <!-- Upload Form -->
            <form action="{{ route('contacts.import') }}" method="POST" enctype="multipart/form-data" id="import-form">
                @csrf
                
                <div class="file-upload-area" id="file-upload-area">
                    <div class="upload-icon">
                        <i class="fas fa-cloud-upload-alt"></i>
                    </div>
                    <h5>Drag & Drop your CSV file here</h5>
                    <p class="text-muted">or click to browse files</p>
                    <input type="file" name="csv_file" id="csv_file" accept=".csv,.txt" class="d-none" required>
                    <div id="file-name"></div>
                </div>

                <div class="progress-container" id="progress-container">
                    <div class="progress">
                        <div class="progress-bar progress-bar-striped progress-bar-animated" 
                             role="progressbar" style="width: 0%"></div>
                    </div>
                </div>

                <!-- Options -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="form-check" style="display: none">
                            <input class="form-check-input" type="checkbox" name="clear_existing" id="clear_existing">
                            <label class="form-check-label" for="clear_existing">
                                <i class="fas fa-trash text-warning"></i>
                                Clear existing contacts before import
                            </label>
                        </div>
                    </div>
                    <div class="col-md-6 text-end">
                        <small class="text-muted">Max file size: 2MB</small>
                    </div>
                </div>

                <!-- Actions -->
                <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                    <button type="button" class="btn btn-outline-danger me-md-2" onclick="clearAllContacts()" 
                            {{ $totalContacts == 0 ? 'disabled' : '' }}>
                        <i class="fas fa-trash-alt"></i> Clear All Contacts
                    </button>
                    <button type="submit" class="btn btn-primary btn-lg" id="import-btn" disabled>
                        <i class="fas fa-upload"></i> Import Contacts
                    </button>
                </div>
            </form>
                </div>

                <!-- Single Contact Tab -->
                <div class="tab-pane fade" id="single-contact" role="tabpanel">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title"><i class="fas fa-user-plus text-primary"></i> Add Single Contact</h5>
                            <p class="text-muted mb-4">Enter contact details manually to add a single contact</p>
                            
                            <form action="{{ route('contacts.store-single') }}" method="POST" id="single-contact-form">
                                @csrf
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="name" class="form-label">Full Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="phone_no" class="form-label">Phone Number <span class="text-danger">*</span></label>
                                        <input type="tel" class="form-control" id="phone_no" name="phone_no" placeholder="0712345678" required>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="socials" class="form-label">Social Media Handle</label>
                                        <input type="text" class="form-control" id="socials" name="socials" placeholder="@username">
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="school_name" class="form-label">School Name</label>
                                        <input type="text" class="form-control" id="school_name" name="school_name" placeholder="University/School name">
                                    </div>
                                </div>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="fas fa-user-plus"></i> Add Contact
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Back Link -->
            <div class="text-center mt-4">
                <a href="{{ route('sms.index') }}" class="text-decoration-none">
                    <i class="fas fa-arrow-left"></i> Back to SMS
                </a>
            </div>
        </div>
    </div>

    <!-- Clear All Modal -->
    <div class="modal fade" id="clearModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Confirm Clear All</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to clear all <strong>{{ number_format($totalContacts) }}</strong> contacts?</p>
                    <p class="text-danger"><i class="fas fa-exclamation-triangle"></i> This action cannot be undone!</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <form action="{{ route('contacts.clear-all') }}" method="POST" class="d-inline">
                        @csrf
                        @method('DELETE')
                        <button type="submit" class="btn btn-danger">Clear All Contacts</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        const fileUploadArea = document.getElementById('file-upload-area');
        const csvFileInput = document.getElementById('csv_file');
        const importBtn = document.getElementById('import-btn');
        const fileNameDiv = document.getElementById('file-name');
        const progressContainer = document.getElementById('progress-container');
        const importForm = document.getElementById('import-form');

        // File upload area click
        fileUploadArea.addEventListener('click', () => csvFileInput.click());

        // File selection
        csvFileInput.addEventListener('change', handleFileSelect);

        // Drag and drop
        fileUploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            fileUploadArea.classList.add('dragover');
        });

        fileUploadArea.addEventListener('dragleave', () => {
            fileUploadArea.classList.remove('dragover');
        });

        fileUploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            fileUploadArea.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                csvFileInput.files = files;
                handleFileSelect();
            }
        });

        function handleFileSelect() {
            const file = csvFileInput.files[0];
            if (file) {
                fileNameDiv.innerHTML = `<i class="fas fa-file-csv text-success"></i> ${file.name} (${formatFileSize(file.size)})`;
                importBtn.disabled = false;
            } else {
                fileNameDiv.innerHTML = '';
                importBtn.disabled = true;
            }
        }

        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        // Form submission with progress
        importForm.addEventListener('submit', function(e) {
            importBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Importing...';
            importBtn.disabled = true;
            progressContainer.style.display = 'block';
            
            // Simulate progress
            let progress = 0;
            const interval = setInterval(() => {
                progress += Math.random() * 15;
                if (progress > 90) progress = 90;
                document.querySelector('.progress-bar').style.width = progress + '%';
            }, 500);

            // Clear interval after form submission
            setTimeout(() => clearInterval(interval), 10000);
        });

        function clearAllContacts() {
            const modal = new bootstrap.Modal(document.getElementById('clearModal'));
            modal.show();
        }
    </script>
</body>
</html>