<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\TxnLog;
use App\Models\Ticket;
use Illuminate\Support\Facades\Log;

class TxnLogSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $logFile = storage_path('logs/laravel.log');

        if (!file_exists($logFile)) {
            $this->command->error('Log file not found: ' . $logFile);
            return;
        }

        $logContent = file_get_contents($logFile);

        // Parse different types of log entries
        $ticketAttempts = $this->parseTicketCreationAttempts($logContent);
        $stkResponses = $this->parseStkResponses($logContent);
        $mpesaCallbacks = $this->parseMpesaCallbacks($logContent);

        $this->command->info('Found ' . count($ticketAttempts) . ' ticket creation attempts');
        $this->command->info('Found ' . count($stkResponses) . ' STK responses');
        $this->command->info('Found ' . count($mpesaCallbacks) . ' M-Pesa callbacks');

        // Create TxnLog records
        $this->createTxnLogRecords($ticketAttempts, $stkResponses, $mpesaCallbacks);
    }

    /**
     * Parse ticket creation attempts from log
     */
    private function parseTicketCreationAttempts($logContent): array
    {
        $attempts = [];

        // Match ticket creation attempts
        preg_match_all('/\[([^\]]+)\] live\.INFO: Ticket creation attempt \{"email":"([^"]+)","phone":"([^"]+)","event_id":"([^"]+)","timestamp":"([^"]+)"\}/', $logContent, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $attempts[] = [
                'log_timestamp' => $match[1],
                'email' => $match[2],
                'phone' => $match[3],
                'event_id' => $match[4],
                'timestamp' => $match[5],
            ];
        }

        return $attempts;
    }

    /**
     * Parse STK responses from log
     */
    private function parseStkResponses($logContent): array
    {
        $responses = [];

        // Match STK response arrays
        preg_match_all('/\[([^\]]+)\] live\.INFO: array \(([\s\S]*?)\)/', $logContent, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $logTimestamp = $match[1];
            $responseData = $match[2];

            // Parse the array data
            $parsedData = $this->parseArrayData($responseData);

            if ($parsedData) {
                $responses[] = [
                    'log_timestamp' => $logTimestamp,
                    'data' => $parsedData,
                ];
            }
        }

        return $responses;
    }

    /**
     * Parse M-Pesa callbacks from log
     */
    private function parseMpesaCallbacks($logContent): array
    {
        $callbacks = [];

        // Match M-Pesa confirmation callbacks
        preg_match_all('/\[([^\]]+)\] live\.INFO: M-Pesa Confirmation Callback \{"data":(\{.*?\})\}/', $logContent, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $logTimestamp = $match[1];
            $callbackData = json_decode($match[2], true);

            if ($callbackData) {
                $callbacks[] = [
                    'log_timestamp' => $logTimestamp,
                    'data' => $callbackData,
                ];
            }
        }

        return $callbacks;
    }

    /**
     * Parse array data from STK response
     */
    private function parseArrayData($arrayString): ?array
    {
        $data = [];

        // Match key-value pairs in the array
        preg_match_all("/'([^']+)' => '([^']*)'/", $arrayString, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $key = $match[1];
            $value = $match[2];
            $data[$key] = $value;
        }

        return !empty($data) ? $data : null;
    }

    /**
     * Create TxnLog records by matching data
     */
    private function createTxnLogRecords($ticketAttempts, $stkResponses, $mpesaCallbacks): void
    {
        $created = 0;
        $skipped = 0;
        $processedTransIds = [];

        foreach ($ticketAttempts as $attempt) {
            // Find corresponding STK response within 30 seconds
            $stkResponse = $this->findStkResponse($stkResponses, $attempt['timestamp'], 30);

            // Find corresponding M-Pesa callback within 5 minutes
            $mpesaCallback = $this->findMpesaCallback($mpesaCallbacks, $attempt['timestamp'], 300);

            // Get ticket details from database
            $ticket = Ticket::where('email', $attempt['email'])->first();

            // Determine status
            $status = $this->determineStatus($stkResponse, $mpesaCallback);

            // Use ticket data if available, otherwise use log data
            $amount = $ticket ? $ticket->total_amount : 5.00; // Default amount for testing
            $ticketReference = $ticket ? $ticket->reference : 'TKT-' . substr($attempt['timestamp'], 0, 8) . '-' . strtoupper(substr(uniqid(), -6));

            // Generate or get trans_id
            $transId = $mpesaCallback['data']['TransID'] ?? $this->generateRandomTransId();

            // Check if we've already processed this trans_id
            if (in_array($transId, $processedTransIds)) {
                $transId = $this->generateRandomTransId(); // Generate unique one
            }
            $processedTransIds[] = $transId;

            // Create TxnLog record using updateOrCreate to handle duplicates
            $txnLogData = [
                'transaction_type' => 'stk',
                'trans_id' => $transId,
                'trans_time' => $mpesaCallback['data']['TransTime'] ?? $attempt['timestamp'],
                'trans_amount' => $amount,
                'ticket_reference_number' => $ticketReference,
                'stk_request' => null, // Not available in logs
                'stk_response' => $stkResponse ? json_encode($stkResponse['data']) : null,
                'amount' => $amount,
                'transaction_date' => $attempt['timestamp'],
                'stk_status' => $status,
                'checkout_request_id' => $stkResponse['data']['CheckoutRequestID'] ?? null,
            ];

            TxnLog::updateOrCreate(
                ['trans_id' => $transId],
                $txnLogData
            );
            $created++;
        }

        $this->command->info("Created {$created} TxnLog records");
        $this->command->info("Skipped {$skipped} records");
    }

    /**
     * Find STK response within time window
     */
    private function findStkResponse($stkResponses, $attemptTimestamp, $windowSeconds): ?array
    {
        $attemptTime = strtotime($attemptTimestamp);

        foreach ($stkResponses as $response) {
            $responseTime = strtotime($response['log_timestamp']);
            $timeDiff = abs($responseTime - $attemptTime);

            if ($timeDiff <= $windowSeconds) {
                return $response;
            }
        }

        return null;
    }

    /**
     * Find M-Pesa callback within time window
     */
    private function findMpesaCallback($mpesaCallbacks, $attemptTimestamp, $windowSeconds): ?array
    {
        $attemptTime = strtotime($attemptTimestamp);

        foreach ($mpesaCallbacks as $callback) {
            $callbackTime = strtotime($callback['log_timestamp']);
            $timeDiff = abs($callbackTime - $attemptTime);

            if ($timeDiff <= $windowSeconds) {
                return $callback;
            }
        }

        return null;
    }

    /**
     * Determine STK status based on available data
     */
    private function determineStatus($stkResponse, $mpesaCallback): string
    {
        if ($mpesaCallback) {
            return 'Launched'; // Use 'Launched' instead of 'Completed' since it's not in enum
        }

        if (!$stkResponse) {
            return 'Launched';
        }

        $responseCode = $stkResponse['data']['ResponseCode'] ?? '';

        switch ($responseCode) {
            case '0':
                return 'Launched';
            case '1':
                return 'Failed';
            case '2':
                return 'Canceled';
            case '3':
                return 'Timeout';
            default:
                return 'Failed';
        }
    }

    /**
     * Generate random transaction ID
     */
    private function generateRandomTransId(): string
    {
        return 'TFM' . strtoupper(substr(uniqid(), -8));
    }
}
