<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\ComplimentaryCode;
use App\Models\ComplimentaryTicket;
use App\Models\Event;
use Illuminate\Support\Str;

class ComplimentaryTicketsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get the first event (or use your highlighted event)
        $event = Event::first();

        if (!$event) {
            $this->command->error('No events found. Please create an event first.');
            return;
        }

        $this->command->info("Seeding complimentary tickets for event: {$event->name}");

        // Create Normal Complimentary Codes
        $normalCodes = [
            [
                'recipient_name' => 'John Doe',
                'recipient_phone' => '254712345678',
                'recipient_email' => 'john.doe@example.com',
                'quantity' => 2,
            ],
            [
                'recipient_name' => 'Jane Smith',
                'recipient_phone' => '254723456789',
                'recipient_email' => 'jane.smith@example.com',
                'quantity' => 1,
            ],
            [
                'recipient_name' => 'Mike Johnson',
                'recipient_phone' => '254734567890',
                'recipient_email' => 'mike.johnson@example.com',
                'quantity' => 3,
            ],
        ];

        // Create VIP Complimentary Codes
        $vipCodes = [
            [
                'recipient_name' => 'Sarah Williams',
                'recipient_phone' => '254745678901',
                'recipient_email' => 'sarah.williams@example.com',
                'quantity' => 1,
            ],
            [
                'recipient_name' => 'David Brown',
                'recipient_phone' => '254756789012',
                'recipient_email' => 'david.brown@example.com',
                'quantity' => 2,
            ],
            [
                'recipient_name' => 'Emily Davis',
                'recipient_phone' => '254767890123',
                'recipient_email' => 'emily.davis@example.com',
                'quantity' => 1,
            ],
        ];

        // Seed Normal Complimentary Codes and Tickets
        $this->command->info('Creating Normal Complimentary tickets...');
        foreach ($normalCodes as $data) {
            $code = ComplimentaryCode::create([
                'event_id' => $event->id,
                'code' => 'COMP-' . strtoupper(Str::random(6)),
                'expires_at' => now()->addMonths(2),
                'generated_by_name' => 'System Seeder',
                'recipient_name' => $data['recipient_name'],
                'recipient_phone' => $data['recipient_phone'],
                'recipient_email' => $data['recipient_email'],
                'ticket_quantity' => $data['quantity'],
                'ticket_type' => 'normal',
                'sms_status' => 'sent',
                'email_status' => 'sent',
                'redeemed_count' => 1, // Mark as partially redeemed
            ]);

            // Create a redeemed ticket for this code
            ComplimentaryTicket::create([
                'event_id' => $event->id,
                'first_name' => explode(' ', $data['recipient_name'])[0],
                'last_name' => explode(' ', $data['recipient_name'])[1] ?? '',
                'phone' => $data['recipient_phone'],
                'email' => $data['recipient_email'],
                'school' => 'Test School',
                'quantity' => 1,
                'reference' => 'COMP-' . Str::uuid(),
                'complimentary_code' => $code->code,
                'ticket_type' => 'normal',
                'is_scanned' => false,
                'scanned_at' => null,
                'scanned_by' => null,
            ]);

            $this->command->info("✓ Created Normal ticket for {$data['recipient_name']} - Code: {$code->code}");
        }

        // Seed VIP Complimentary Codes and Tickets
        $this->command->info('Creating VIP Complimentary tickets...');
        foreach ($vipCodes as $data) {
            $code = ComplimentaryCode::create([
                'event_id' => $event->id,
                'code' => 'VIP-' . strtoupper(Str::random(6)),
                'expires_at' => now()->addMonths(2),
                'generated_by_name' => 'System Seeder',
                'recipient_name' => $data['recipient_name'],
                'recipient_phone' => $data['recipient_phone'],
                'recipient_email' => $data['recipient_email'],
                'ticket_quantity' => $data['quantity'],
                'ticket_type' => 'vip',
                'sms_status' => 'sent',
                'email_status' => 'sent',
                'redeemed_count' => 1, // Mark as partially redeemed
            ]);

            // Create a redeemed ticket for this code
            ComplimentaryTicket::create([
                'event_id' => $event->id,
                'first_name' => explode(' ', $data['recipient_name'])[0],
                'last_name' => explode(' ', $data['recipient_name'])[1] ?? '',
                'phone' => $data['recipient_phone'],
                'email' => $data['recipient_email'],
                'school' => 'VIP Academy',
                'quantity' => 1,
                'reference' => 'VIP-' . Str::uuid(),
                'complimentary_code' => $code->code,
                'ticket_type' => 'vip',
                'is_scanned' => false,
                'scanned_at' => null,
                'scanned_by' => null,
            ]);

            $this->command->info("✓ Created VIP ticket for {$data['recipient_name']} - Code: {$code->code}");
        }

        $this->command->info('');
        $this->command->info('✅ Seeding completed successfully!');
        $this->command->info('');
        $this->command->table(
            ['Type', 'Count'],
            [
                ['Normal Complimentary Codes', count($normalCodes)],
                ['VIP Complimentary Codes', count($vipCodes)],
                ['Total Tickets Created', count($normalCodes) + count($vipCodes)],
            ]
        );

        // Display some ticket references for testing
        $this->command->info('');
        $this->command->info('Sample ticket references for testing:');
        $normalTicket = ComplimentaryTicket::where('ticket_type', 'normal')->first();
        $vipTicket = ComplimentaryTicket::where('ticket_type', 'vip')->first();

        if ($normalTicket) {
            $this->command->info("Normal: " . url("/get/complimentary-ticket/{$normalTicket->reference}"));
        }
        if ($vipTicket) {
            $this->command->info("VIP: " . url("/get/complimentary-ticket/{$vipTicket->reference}"));
        }
    }
}
