<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class SmsLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'contact_name',
        'phone_number',
        'school_name',
        'message_content',
        'status',
        'response_data',
        'message_id',
        'cost',
        'parts',
        'sent_at',
        'campaign_name',
    ];

    protected $casts = [
        'sent_at' => 'datetime',
        'response_data' => 'array',
        'cost' => 'decimal:2',
    ];

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeForCampaign($query, $campaignName)
    {
        return $query->where('campaign_name', $campaignName);
    }

    // Helper methods
    public function markAsSent($messageId = null, $cost = null)
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => Carbon::now(),
            'message_id' => $messageId,
            'cost' => $cost,
        ]);
    }

    public function markAsFailed($responseData = null)
    {
        $this->update([
            'status' => 'failed',
            'response_data' => $responseData,
        ]);
    }

    public function getFormattedCostAttribute()
    {
        return $this->cost ? 'KSH ' . number_format($this->cost, 2) : 'N/A';
    }

    public function getStatusBadgeAttribute()
    {
        return match ($this->status) {
            'sent' => '<span class="badge bg-success">Sent</span>',
            'failed' => '<span class="badge bg-danger">Failed</span>',
            'pending' => '<span class="badge bg-warning">Pending</span>',
            default => '<span class="badge bg-secondary">Unknown</span>',
        };
    }
}
