<?php

namespace App\Http\Controllers;

use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function __construct()
        {
            // Default: super-admin|admin can access most actions
            $this->middleware('role:super-admin|admin')->except(['destroy', 'scanner']);

            // Only super-admin can delete users
            $this->middleware('role:super-admin')->only(['destroy']);

            // Special rule for scanner role: ONLY allow access to scanner() method
            $this->middleware(function ($request, $next) {
                if (auth()->check() && auth()->user()->hasRole('scanner')) {
                    // Allow only the scanner route/method
                    if ($request->route()->getName() === 'users.scanner' || 
                        $request->is('login') || 
                        $request->route()->getName() === 'login') {
                        return $next($request);
                    }

                    // Optionally redirect scanners trying to access other pages
                    return redirect()->route('users.scanner')
                        ->with('error', 'You do not have permission to access this page.');
                }

                return $next($request);
            });
        }

    public function index()
    {
        $users = User::with('roles')->paginate(10); // 10 users per page
        return view('users.index', compact('users'));
    }

    public function show(User $user)
    {
        $user->load('roles'); // Eager load roles for the user
        return view('users.show', compact('user'));
    }

    public function create()
    {
        $roles = Role::all();
        return view('me.new', compact('roles'));
    }

    public function store(Request $request)
            {
                $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|unique:users,phone',
            'email' => 'nullable|email|unique:users,email', // nullable + unique if provided
            'password' => 'required|string|min:8',
            'roles' => 'required|array',
        ]);
        // Clean and format Kenyan phone number
        $phone = $this->formatKenyanPhone($request->phone);

        $user = User::create([
            'name' => $request->name,
            'phone' => $phone,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        $user->syncRoles($request->roles);

        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }

    /**
     * Format Kenyan phone number to 254712345678 format
     */
    private function formatKenyanPhone($phone)
    {
        // Remove all non-digit characters except leading +
        $phone = preg_replace('/[^\d+]/', '', $phone);
        
        // Remove any leading zeros
        $phone = ltrim($phone, '0');
        
        // Handle +254 prefix
        if (str_starts_with($phone, '+254')) {
            $phone = '254' . substr($phone, 4);
        }
        
        // Handle 254 prefix (already correct)
        if (str_starts_with($phone, '254')) {
            return $phone;
        }
        
        // Handle landlines starting with 011 (Nairobi area code)
        if (str_starts_with($phone, '11')) {
            // Return as 25411... format for landlines
            return '254' . $phone;
        }
        
        // Handle 07... format - convert to 2547...
        if (str_starts_with($phone, '7') && strlen($phone) === 9) {
            return '254' . $phone;
        }
        
        // Default case - assume it's already in correct format or add 254
        if (strlen($phone) === 9 && is_numeric($phone)) {
            return '254' . $phone;
        }
        
        return $phone;
    }

    public function edit(User $user)
    {
        $roles = Role::all();
        return view('users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|unique:users,phone,' . $user->id,
            'email' => 'nullable|email|unique:users,email,' . $user->id, // Changed to nullable
            'password' => 'nullable|string|min:8',
            'roles' => 'required|array',
        ]);

        // Clean and format Kenyan phone number
        $phone = $this->formatKenyanPhone($request->phone);

        $user->update([
            'name' => $request->name,
            'phone' => $phone,
            'email' => $request->email,
            'password' => $request->password ? Hash::make($request->password) : $user->password,
        ]);

        $user->syncRoles($request->roles);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    public function scanner()
    {
        $user = auth()->user();
        return view('users.scanner', compact('user'));
    }
}