<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class StkTransactionController extends Controller
{
    /**
     * Display STK transactions web interface
     */
    public function index()
    {
        return view('transactions.stk-index');
    }

    /**
     * Log STK transaction
     */
    public function logStkTransaction(Request $request)
    {
        try {
            $data = $request->validate([
                'ticket_reference_number' => 'required|string',
                'stk_request' => 'required|string',
                'stk_response' => 'required|string',
                'amount' => 'required|numeric',
                'msisdn' => 'required|string',
                'transaction_type' => 'required|string',
            ]);

            // Parse STK response to extract CheckoutRequestID
            $stkResponse = json_decode($data['stk_response'], true);
            $checkoutRequestId = $stkResponse['CheckoutRequestID'] ?? null;

            // Determine STK status based on response
            $stkStatus = $this->determineStkStatus($stkResponse);

            $transaction = Transaction::create([
                'transaction_type' => $data['transaction_type'],
                'trans_id' => $this->generateTransactionId(),
                'ticket_reference_number' => $data['ticket_reference_number'],
                'stk_request' => $data['stk_request'],
                'stk_response' => $data['stk_response'],
                'amount' => $data['amount'],
                'transaction_date' => now(),
                'stk_status' => $stkStatus,
                'checkout_request_id' => $checkoutRequestId,
                'msisdn' => $data['msisdn'],
                'trans_time' => now()->format('YmdHis'),
                'trans_amount' => $data['amount'],
            ]);

            Log::info('STK Transaction logged successfully', [
                'transaction_id' => $transaction->id,
                'ticket_reference' => $data['ticket_reference_number'],
                'amount' => $data['amount'],
                'status' => $stkStatus,
                'checkout_request_id' => $checkoutRequestId
            ]);

            return response()->json([
                'success' => true,
                'message' => 'STK transaction logged successfully',
                'transaction' => $transaction
            ], 201);
        } catch (\Exception $e) {
            Log::error('Error logging STK transaction', [
                'error' => $e->getMessage(),
                'data' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error logging STK transaction',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper method to log STK transaction from other controllers
     */
    public static function logStkTransactionHelper($ticketReference, $stkRequest, $stkResponse, $amount, $msisdn, $transactionType = 'STK')
    {
        try {
            $stkResponseArray = is_array($stkResponse) ? $stkResponse : json_decode($stkResponse, true);
            $checkoutRequestId = $stkResponseArray['CheckoutRequestID'] ?? null;
            $stkStatus = self::determineStkStatusHelper($stkResponseArray);

            $transaction = Transaction::create([
                'transaction_type' => $transactionType,
                'trans_id' => 'STK_' . time() . '_' . rand(1000, 9999),
                'ticket_reference_number' => $ticketReference,
                'stk_request' => is_string($stkRequest) ? $stkRequest : json_encode($stkRequest),
                'stk_response' => is_string($stkResponse) ? $stkResponse : json_encode($stkResponse),
                'amount' => $amount,
                'transaction_date' => now(),
                'stk_status' => $stkStatus,
                'checkout_request_id' => $checkoutRequestId,
                'msisdn' => $msisdn,
                'trans_time' => now()->format('YmdHis'),
                'trans_amount' => $amount,
            ]);

            Log::info('STK Transaction logged via helper', [
                'transaction_id' => $transaction->id,
                'ticket_reference' => $ticketReference,
                'amount' => $amount,
                'status' => $stkStatus,
                'checkout_request_id' => $checkoutRequestId
            ]);

            return $transaction;
        } catch (\Exception $e) {
            Log::error('Error logging STK transaction via helper', [
                'error' => $e->getMessage(),
                'ticket_reference' => $ticketReference,
                'amount' => $amount
            ]);

            return null;
        }
    }

    /**
     * Update STK transaction status
     */
    public function updateStkStatus(Request $request, $transactionId)
    {
        try {
            $data = $request->validate([
                'stk_status' => 'required',
                'stk_response' => 'nullable|string',
            ]);

            $transaction = Transaction::findOrFail($transactionId);

            $transaction->update([
                'stk_status' => $data['stk_status'],
                'stk_response' => $data['stk_response'] ?? $transaction->stk_response,
            ]);

            Log::info('STK Transaction status updated', [
                'transaction_id' => $transaction->id,
                'new_status' => $data['stk_status'],
                'ticket_reference' => $transaction->ticket_reference_number
            ]);

            return response()->json([
                'success' => true,
                'message' => 'STK transaction status updated successfully',
                'transaction' => $transaction
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating STK transaction status', [
                'error' => $e->getMessage(),
                'transaction_id' => $transactionId
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error updating STK transaction status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get STK transactions
     */
    public function getStkTransactions(Request $request)
    {
        try {
            $query = Transaction::whereNotNull('ticket_reference_number');

            // Filter by status
            if ($request->has('status')) {
                $query->where('stk_status', $request->status);
            }

            // Filter by date range
            if ($request->has('start_date')) {
                $query->whereDate('transaction_date', '>=', $request->start_date);
            }

            if ($request->has('end_date')) {
                $query->whereDate('transaction_date', '<=', $request->end_date);
            }

            // Filter by ticket reference
            if ($request->has('ticket_reference')) {
                $query->where('ticket_reference_number', 'like', '%' . $request->ticket_reference . '%');
            }

            $transactions = $query->orderBy('transaction_date', 'desc')->paginate(20);

            return response()->json([
                'success' => true,
                'transactions' => $transactions
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching STK transactions', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching STK transactions',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get STK transaction statistics
     */
    public function getStkStatistics()
    {
        try {
            $statistics = [
                'total_transactions' => Transaction::whereNotNull('ticket_reference_number')->count(),
                'launched' => Transaction::where('stk_status', 'Launched')->count(),
                'failed' => Transaction::where('stk_status', 'Failed')->count(),
                'canceled' => Transaction::where('stk_status', 'Canceled')->count(),
                'timeout' => Transaction::where('stk_status', 'Timeout')->count(),
                'success' => Transaction::where('stk_status', 'Success')->count(),
                'invalid_user_input' => Transaction::where('stk_status', 'Invalid-user-input')->count(),
                'total_amount' => Transaction::whereNotNull('ticket_reference_number')->sum('amount'),
                'success_rate' => $this->calculateSuccessRate(),
            ];

            return response()->json([
                'success' => true,
                'statistics' => $statistics
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching STK statistics', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching STK statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Determine STK status based on response
     */
    private function determineStkStatus($stkResponse)
    {
        if (!is_array($stkResponse)) {
            return 'Failed';
        }

        $responseCode = $stkResponse['ResponseCode'] ?? null;
        $responseDescription = $stkResponse['ResponseDescription'] ?? '';
        $resultCode = $stkResponse['ResultCode'] ?? null;
        $resultDesc = $stkResponse['ResultDesc'] ?? '';


        // Check for specific error conditions
        if (str_contains(strtolower($resultDesc), 'invalid user input')) {
            return 'Invalid-user-input';
        }

        if (str_contains(strtolower($resultDesc), 'cancel') || str_contains(strtolower($resultDesc), 'user pressed cancel')) {
            return 'Canceled';
        }
        if (str_contains(strtolower($resultDesc), 'no response from user')) {
            return 'No-response-from-user';
        }

        if (str_contains(strtolower($resultDesc), 'Cancelled by user')) {
            return 'Canceled';
        }

        if (str_contains(strtolower($resultDesc), 'timeout') || str_contains(strtolower($resultDesc), 'network failure')) {
            return 'Timeout';
        }

        if (str_contains(strtolower($resultDesc), 'user cannot be reached')) {
            return 'Timeout';
        }

        if (str_contains(strtolower($resultDesc), 'information is invalid') || str_contains(strtolower($resultDesc), 'network failure')) {
            return 'Failed';
        }

        // Check for API errors
        if (isset($stkResponse['errorCode']) && $stkResponse['errorCode'] === '500.001.1001') {
            $errorMessage = $stkResponse['errorMessage'] ?? '';
            if (str_contains(strtolower($errorMessage), 'invalid user input')) {
                return 'Invalid-user-input';
            }
            if (str_contains(strtolower($errorMessage), 'cancel')) {
                return 'Canceled';
            }
            if (str_contains(strtolower($errorMessage), 'network failure')) {
                return 'Timeout';
            }
        }

         // Check for success
         if ($responseCode === '0' && $resultCode === '0') {
            return 'Success';
        }

        // Default to failed
        return 'Failed';
    }

    /**
     * Helper method to determine STK status
     */
    private static function determineStkStatusHelper($stkResponse)
    {
        if (!is_array($stkResponse)) {
            return 'Failed';
        }

        $responseCode = $stkResponse['ResponseCode'] ?? null;
        $responseDescription = $stkResponse['ResponseDescription'] ?? '';
        $resultCode = $stkResponse['ResultCode'] ?? null;
        $resultDesc = $stkResponse['ResultDesc'] ?? '';

       

        // Check for specific error conditions
        if (str_contains(strtolower($resultDesc), 'invalid user input')) {
            return 'Invalid-user-input';
        }

        if (str_contains(strtolower($resultDesc), 'cancel') || str_contains(strtolower($resultDesc), 'user pressed cancel')) {
            return 'Canceled';
        }

        if (str_contains(strtolower($resultDesc), 'timeout') || str_contains(strtolower($resultDesc), 'network failure')) {
            return 'Timeout';
        }

        if (str_contains(strtolower($resultDesc), 'user cannot be reached')) {
            return 'Timeout';
        }

        if (str_contains(strtolower($resultDesc), 'information is invalid') || str_contains(strtolower($resultDesc), 'network failure')) {
            return 'Failed';
        }

        if (str_contains(strtolower($resultDesc), 'no response from user')) {
    
            return 'No-response-from-user';
        }


        if (str_contains(strtolower($resultDesc), 'Cancelled by user')) {
            return 'Canceled';
        }

        // Check for API errors
        if (isset($stkResponse['errorCode']) && $stkResponse['errorCode'] === '500.001.1001') {
            $errorMessage = $stkResponse['errorMessage'] ?? '';
            if (str_contains(strtolower($errorMessage), 'invalid user input')) {
                return 'Invalid-user-input';
            }
            if (str_contains(strtolower($errorMessage), 'cancel')) {
                return 'Canceled';
            }
            if (str_contains(strtolower($errorMessage), 'network failure')) {
                return 'Timeout';
            }
        }

         // Check for success
         if ($responseCode === '0' && $resultCode === '0') {
            return 'Success';
        }

        // Default to failed
        return 'Failed';
    }

    /**
     * Generate unique transaction ID
     */
    private function generateTransactionId()
    {
        return 'STK_' . time() . '_' . rand(1000, 9999);
    }

    /**
     * Calculate success rate
     */
    private function calculateSuccessRate()
    {
        $total = Transaction::whereNotNull('ticket_reference_number')->count();
        $launched = Transaction::where('stk_status', 'Launched')->count();

        if ($total === 0) {
            return 0;
        }

        return round(($launched / $total) * 100, 2);
    }
}
