<?php

namespace App\Http\Controllers;

use App\Models\MessageTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class MessageTemplateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $templates = MessageTemplate::orderBy('created_at', 'desc')->get();
        return view('message-templates.index', compact('templates'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('message-templates.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:message_templates,name',
            'content' => 'required|string',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        MessageTemplate::create($request->only(['name', 'content', 'description']) + [
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.message-templates.index')
            ->with('success', 'Message template created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(MessageTemplate $messageTemplate)
    {
        return view('message-templates.show', compact('messageTemplate'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MessageTemplate $messageTemplate)
    {
        return view('message-templates.edit', compact('messageTemplate'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MessageTemplate $messageTemplate)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:message_templates,name,' . $messageTemplate->getKey(),
            'content' => 'required|string',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $messageTemplate->update($request->only(['name', 'content', 'description']) + [
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.message-templates.index')
            ->with('success', 'Message template updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MessageTemplate $messageTemplate)
    {
        $messageTemplate->delete();
        return redirect()->route('admin.message-templates.index')
            ->with('success', 'Message template deleted successfully!');
    }

    /**
     * Get templates for AJAX requests
     */
    public function getTemplates()
    {
        $templates = MessageTemplate::active()->select('id', 'name', 'content')->get();
        return response()->json($templates);
    }
}
