<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Traits\C2BSetup;
use Illuminate\Support\Facades\Log;
use App\Models\Event;
use App\Models\Ticket;
use App\Models\TxnLog;
use App\Models\Log as LogModel;
use App\Models\Transaction;
use App\Models\ComplimentaryTicket;
use GuzzleHttp\Client;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;


class Homecontroller extends Controller
{
    use C2BSetup;

    public function home()
    {
        // Get all events (if needed in future) and the event with id = 1
        $events = Event::all();
        $highlightedEvent = Event::find(1); // or use firstWhere('id', 1);

        return view('welcome', compact('events', 'highlightedEvent'));
    }

    public function index($slug)
    {
        $event = Event::where('slug', $slug)->firstOrFail();
        $highlightedEvent = Event::find(1);

        if ($event->slug !== $slug) {
            return redirect()->route('ticket', $event->slug);
        }

        return view('ticket.index', [
            'event' => $event,
            'highlightedEvent' => $highlightedEvent
        ]);
    }

    public function contact()
    {
        $highlightedEvent = Event::find(1);
        return view('contact', compact('highlightedEvent'));
    }
    public function gateMobile()
    {
        $highlightedEvent = Event::find(1);
        $event = $highlightedEvent ?? Event::first();
        return view('ticket.gate-mobile', [
            'event' => $event,
            'highlightedEvent' => $highlightedEvent
        ]);
    }
    public function Ticket(Request $request)
    {
        // Log the ticket creation attempt
        Log::info('Ticket creation attempt', [
            'email' => $request->input('email'),
            'phone' => $request->input('phone'),
            'event_id' => $request->input('event_id'),
            'timestamp' => now()->toDateTimeString()
        ]);

        // Validate the request (add ticket_type validation)
        $validated = $request->validate([
            'firstName' => 'required|string|max:255',
            'lastName' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'school' => 'required|string|max:255',
            'ticket_type' => 'required|string|in:early_bird,group_4,group_5,group_10,gate',
            'ticket_quantity' => 'required|integer|min:1',
            'event_id' => 'required|integer|exists:events,id',
        ]);

        // Check for duplicate submissions within the last 1minutes
        // BUT only for tickets that are not failed - allow retries for failed payments
        $recentTicket = Ticket::where('email', $validated['email'])
            ->where('phone', $validated['phone'])
            ->where('event_id', $validated['event_id'])
            ->where('created_at', '>=', now()->subMinutes(1))
            ->where('payment_status', 'pending')
            ->first();

        if ($recentTicket) {
            Log::warning('Duplicate ticket submission detected', [
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'event_id' => $validated['event_id'],
                'recent_ticket_id' => $recentTicket->id,
                'recent_ticket_created_at' => $recentTicket->created_at,
                'recent_ticket_payment_status' => $recentTicket->payment_status,
                'time_difference_seconds' => now()->diffInSeconds($recentTicket->created_at)
            ]);

            $message = 'A ticket with these details was already submitted recently. Please wait for a minute';
            if ($recentTicket->payment_status === 'completed') {
                $message .= ' Your payment was successful. Please check your email or download your ticket.';
            } else {
                $message .= ' Please wait one  minutes before trying again.';
            }

            return response()->json([
                'status' => 'error',
                'message' => $message,
                'data' => $validated
            ]);
        }

        // Get the event and determine amount based on ticket type
        $event = Event::findOrFail($validated['event_id']);

        // Get price for selected ticket type
        $amount = $event->tickets[$validated['ticket_type']];

        // Calculate the total amount based on the quantity of packages/groups
        $totalAmount = $amount * $validated['ticket_quantity'];

        // Calculate actual number of individual tickets
        $individualTicketsMap = [
            'early_bird' => 1,
            'group_4' => 4,
            'group_5' => 5,
            'group_10' => 10,
            'gate' => 1
        ];
        $individualTicketCount = $individualTicketsMap[$validated['ticket_type']] * $validated['ticket_quantity'];

        // Generate unique reference for this ticket
        $reference = Ticket::generateReference();

        // Create and save the ticket
        $ticket = Ticket::create([
            'event_id' => $validated['event_id'],
            'first_name' => $validated['firstName'],
            'last_name' => $validated['lastName'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'school' => $validated['school'],
            'ticket_type' => $validated['ticket_type'],
            'amount' => $amount,
            'quantity' => $individualTicketCount,
            'total_amount' => $totalAmount,
            'reference' => $reference,
            'payment_status' => 'pending'
        ]);

        $stkResponse = $this->processStkPush($validated['phone'], $totalAmount, $reference);

        if (isset($stkResponse['status']) && $stkResponse['status'] == 'error') {
            // Mark the ticket as failed since the payment request failed
            $ticket->payment_status = 'failed';
            $ticket->save();

            Log::info('Ticket marked as failed due to STK push failure', [
                'ticket_id' => $ticket->id,
                'reference' => $reference,
                'error_message' => $stkResponse['message'] ?? 'Unknown error'
            ]);

            return response()->json([
                'status' => 'error',
                'message' => $stkResponse['message'] ?? 'Payment request failed',
                'data' => $validated
            ]);
        }

        Log::info('STK Push successful', [
            'stk_response' => $stkResponse
        ]);

        $phone_str = '254' . substr($validated['phone'], -9);
        // $telco = $this->telco($phone_str);

        TxnLog::create([
            'transaction_type' => 'stk',
            'trans_id' => uniqid(),
            'ticket_reference_number' => $reference,
            'stk_request' => json_encode(array($validated['phone'], $totalAmount, $reference)),
            'stk_response' => json_encode($stkResponse),
            'amount' => $totalAmount,
            'transaction_date' => now(),
            'stk_status' => 'Launched',
            'checkout_request_id' => $stkResponse['data']['CheckoutRequestID'],
            'msisdn' => $validated['phone'],
            'trans_time' => now()->format('YmdHis'),
            'trans_amount' => $totalAmount,
        ]);

        if (env('APP_ENV') == 'staging') {

            // Send sms to the customer
            $phone = $phone_str;
            // $message = "Confirmed Ksh. {$ticket->total_amount}, you purchased {$ticket->quantity} early bird tickets on " . now()->timezone('Africa/Nairobi')->format('d/m/Y') . " at " . now()->timezone('Africa/Nairobi')->format('H:i') . ", for {$ticket->event->name} at {$ticket->event->venue} on " . \Carbon\Carbon::parse($ticket->event->date)->timezone('Africa/Nairobi')->format('jS M Y') . ". E-Ticket Ref#: {$ticket->reference}. Download your tickets here: " . url("/get/ticket/{$ticket->reference}");
            $message = "Dear {$validated['firstName']} {$validated['lastName']},Welcome to {$event->name}  \nTicket #: {$reference}  \nTickets: {$validated['ticket_quantity']}  \nGet your tickets here:" . url("/get/ticket/{$reference}");

            // Send SMS notification
            $this->sendSms($phone, $message);
            // Send email notification
            $this->sendEmail(
                $ticket->email,
                "Your PACE TeenzFest Ticket Confirmation",
                $ticket,
                $ticket->event
            );
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Payment request sent successfully',
            'data' => [
                'ticket' => $ticket,
                'payment' => $stkResponse['data'] ?? null,
            ]
        ]);
    }


    public function processStkPush($phone, $amount, $reference)
    {
        try {
            // Normalize phone number to 2547XXXXXXXX or 2541XXXXXXXX
            $phone = trim($phone);
            if (strpos($phone, '+') === 0) {
                $phone = substr($phone, 1);
            }
            if (strpos($phone, '254') === 0 && strlen($phone) === 13) {
                // Already in correct format
            } elseif (preg_match('/^(07|01)\d{8}$/', $phone)) {
                $phone = '254' . substr($phone, 1);
            } elseif (preg_match('/^7\d{8}$/', $phone)) {
                $phone = '254' . $phone;
            } elseif (preg_match('/^1\d{8}$/', $phone)) {
                $phone = '254' . $phone;
            }
            // else: leave as is, or add more rules as needed

            $paybill = "4083001";
            $callBackURL = url('/api/v1/c2b-stk-callback');

            $transactionType = "CustomerPayBillOnline";
            $accountReference = $reference;
            $token = $this->generateC2bMpesaToken();

            $stkPushSimulation = $this->stkPush($paybill, $transactionType, $amount, $phone, $paybill, $phone, $callBackURL, $accountReference, null, null, $token);

            $response = json_decode($stkPushSimulation, true);

            Log::info($response);

            $status = 'success';
            $message = 'Success. Request accepted for processing';
            if (empty($response['CheckoutRequestID'])) {
                $status = 'error';
                $message = $response['errorMessage'] ?? 'Payment processing failed';
            }

            $CheckoutRequestID = $response['CheckoutRequestID'] ?? null;
            $MerchantRequestID = $response['MerchantRequestID'] ?? null;

            return [
                'status' => $status,
                'data' => $response,
                'message' => $message,
            ];
        } catch (\Exception $exception) {
            Log::error('STK Push Exception: ' . $exception->getMessage());
            return [
                'status' => 'error',
                'message' => $exception->getMessage(),
                'data' => null
            ];
        }
    }
    //Method to return to event page

    public function validateStkPush(Request $request)
    {
        $checkoutRequestID = $request->input('checkoutRequestID');

        $txnLog = TxnLog::where('checkout_request_id', $checkoutRequestID)
            ->where('stk_status', 'Launched')
            ->first();

        if ($txnLog) {
            // Check if the ticket status is paid - stk_status should be success
            $ticket = Ticket::where('reference', $txnLog->ticket_reference_number)->first();

            if ($ticket && $ticket->payment_status == 'completed') {
                $txnLog->update([
                    'stk_status' => 'Success'
                ]);

                return response()->json([
                    'response' => $ticket,
                    'message' => 'Ticket already paid',
                    'status_updated' => false
                ]);
            }

            // Validate the STK push - FIXED: variable name was wrong
            $validationResult = $this->validateStkPushRequest($checkoutRequestID); // Fixed method call

            if (isset($validationResult['ResponseCode'])) { // Fixed variable name
                $txnLog->update([
                    'stk_status' => 'Success'
                ]);

                if (isset($validationResult['ResultDesc'])) { // Fixed variable name
                    $errorMessage = $validationResult['ResultDesc'] ?? ''; // Fixed variable name

                    if (str_contains(strtolower($errorMessage), 'cancel')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'network failure')) {
                        $stkStatus = 'Timeout';
                    } elseif (str_contains(strtolower($errorMessage), 'invalid user input')) {
                        $stkStatus = 'Invalid-user-input';
                    } elseif (str_contains(strtolower($errorMessage), 'insufficient')) {
                        $stkStatus = 'Insufficient-funds';
                    } elseif (str_contains(strtolower($errorMessage), 'user cannot be reached')) {
                        $stkStatus = 'User-cannot-be-reached';
                    } elseif (str_contains(strtolower($errorMessage), 'response from user')) {
                        $stkStatus = 'No-response-from-user';
                    } elseif (str_contains(strtolower($errorMessage), 'user pressed cancel')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'cancelled by user')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'information is invalid')) {
                        $stkStatus = 'Failed';
                    } elseif (str_contains(strtolower($errorMessage), 'network failure')) {
                        $stkStatus = 'Timeout';
                    } elseif (str_contains(strtolower($errorMessage), 'under processing')) {
                        $stkStatus = 'Launched';
                    } else {
                        $stkStatus = 'Failed';
                    }

                    $txnLog->update([
                        'stk_status' => $stkStatus
                    ]);
                }
            } else {
                if (isset($validationResult['errorCode'])) { // Fixed variable name
                    $errorMessage = $validationResult['errorMessage'] ?? ''; // Fixed variable name

                    if (str_contains(strtolower($errorMessage), 'cancel')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'network failure')) {
                        $stkStatus = 'Timeout';
                    } elseif (str_contains(strtolower($errorMessage), 'invalid user input')) {
                        $stkStatus = 'Invalid-user-input';
                    } elseif (str_contains(strtolower($errorMessage), 'insufficient')) {
                        $stkStatus = 'Insufficient-funds';
                    } elseif (str_contains(strtolower($errorMessage), 'user cannot be reached')) {
                        $stkStatus = 'User-cannot-be-reached';
                    } elseif (str_contains(strtolower($errorMessage), 'response from user')) {
                        $stkStatus = 'No-response-from-user';
                    } elseif (str_contains(strtolower($errorMessage), 'user pressed cancel')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'cancelled by user')) {
                        $stkStatus = 'Canceled';
                    } elseif (str_contains(strtolower($errorMessage), 'information is invalid')) {
                        $stkStatus = 'Failed';
                    } elseif (str_contains(strtolower($errorMessage), 'network failure')) {
                        $stkStatus = 'Timeout';
                    } else {
                        $stkStatus = 'Failed';
                    }

                    $txnLog->update([
                        'stk_status' => $stkStatus
                    ]);
                }
            }

            $oldStatus = $txnLog->stk_status;

            // Log the status update - FIXED syntax errors
            LogModel::create([
                'source' => 'STK Status Update', // Fixed typo and syntax
                'content' => json_encode([
                    'checkout_request_id' => $checkoutRequestID,
                    'old_status' => $oldStatus,
                    'response' => $validationResult // Fixed variable name
                ])
            ]);

            return response()->json([
                'response' => $validationResult, // Fixed variable name
                'message' => 'Status updated successfully',
                'status_updated' => true
            ]);
        }

        return response()->json([
            'response' => 'No status update performed, Status already checked',
            'message' => 'No matching transaction log found',
            'status_updated' => false
        ]);
    }

    /**
     * Helper method to validate STK push request
     * This method should contain the actual STK validation logic
     */
    private function validateStkPushRequest($checkoutRequestID)
    {
        try {
            // Add your STK validation logic here
            // This should make the API call to M-Pesa to check transaction status
            // For now, returning empty array as placeholder
            return [];
        } catch (\Exception $e) {
            Log::error('STK Validation Error: ' . $e->getMessage());
            return ['errorCode' => '500', 'errorMessage' => $e->getMessage()];
        }
    }

    public function CreateEvent(Request $request)
    {
        // return to event page
        $highlightedEvent = Event::find(1);
        return view('events.create-event', compact('highlightedEvent'));
    }
    //Method to create event
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:events', // Ensure event names are unique
            'venue' => 'required|string|max:255',
            'date' => 'required|date',
            'early_bird' => 'required|numeric',
            'advance' => 'required|numeric',
            'gate' => 'required|numeric',
            'event_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'intro_video' => 'nullable|url'
        ]);

        // Handle image upload
        $imagePath = null;
        if ($request->hasFile('event_image')) {
            $image = $request->file('event_image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('uploads/events'), $imageName);
            $imagePath = 'uploads/events/' . $imageName;
        }

        // Create the event with all fields including name and auto-generated slug
        Event::create([
            'name' => $validated['name'],
            'slug' => Str::slug($validated['name']), // Auto-generate slug from name
            'venue' => $validated['venue'],
            'date' => $validated['date'],
            'tickets' => [
                'early_bird' => $validated['early_bird'],
                'advance' => $validated['advance'],
                'gate' => $validated['gate'],
            ],
            'event_image' => $imagePath,
            'intro_video' => $validated['intro_video'] ?? null,
        ]);

        return back()->with('success', 'Event created successfully!');
    }
    //end method.

    public function StkCallback(Request $request)
    {
        try {
            $data = $request->all();

            if (!isset($data['Body']['stkCallback'])) {
                Log::error('Invalid STK callback data format', $data);
                return response()->json([
                    'ResponseCode' => '00000000',
                    'ResponseDesc' => 'Accept service request successfully',
                ]);
            }

            $resultCode = $data['Body']['stkCallback']['ResultCode'];
            $merchant_request_id = $data['Body']['stkCallback']['MerchantRequestID'];
            $checkout_request_id = $data['Body']['stkCallback']['CheckoutRequestID'];
            $resultDesc = $data['Body']['stkCallback']['ResultDesc'];

            if ($resultCode == 0) {
                $amount = 0;
                $mpesaReceiptNumber = '';
                $phone_number = '';
                $customerFirstName = '';

                foreach ($data['Body']['stkCallback']['CallbackMetadata']['Item'] as $item) {
                    if ($item['Name'] === 'Amount') {
                        $amount = $item['Value'];
                    } else if ($item['Name'] === 'MpesaReceiptNumber') {
                        $mpesaReceiptNumber = $item['Value'];
                    } else if ($item['Name'] === 'PhoneNumber') {
                        $phone_number = $item['Value'];
                    } else if ($item['Name'] === 'FirstName') {
                        $customerFirstName = $item['Value'];
                    }
                }

                // Find the ticket by reference (we're assuming the reference is stored in AccountReference)
                $ticket = \App\Models\Ticket::where('reference', $data['Body']['stkCallback']['AccountReference'] ?? '')
                    ->first();

                if ($ticket) {
                    // Update the ticket with payment details
                    $ticket->payment_status = 'completed';
                    $ticket->mpesa_receipt_number = $mpesaReceiptNumber;
                    $ticket->payment_date = now()->timezone('Africa/Nairobi');
                    if (!empty($customerFirstName) && ($ticket->ticket_type === 'gate' || (strtolower($ticket->first_name) === 'gate' && strtolower($ticket->last_name) === 'guest'))) {
                        $ticket->first_name = $customerFirstName;
                        $ticket->last_name = '';
                    }
                    $ticket->save();

                    // Send Discord notification to admin
                    // $this->sendDiscordNotification($ticket);

                    Log::info("Payment completed for ticket: {$ticket->id}, Receipt: {$mpesaReceiptNumber}");
                } else {
                    Log::warning("Ticket not found for reference: " . ($data['Body']['stkCallback']['AccountReference'] ?? 'Unknown'));
                }
            } else {
                Log::error("STK Push failed with code: {$resultCode}, Desc: {$resultDesc}");

                // Find the ticket and mark it as failed
                $ticket = \App\Models\Ticket::where('reference', $data['Body']['stkCallback']['AccountReference'] ?? '')
                    ->first();

                if ($ticket) {
                    $ticket->payment_status = 'failed';
                    $ticket->save();

                    Log::info("Payment failed for ticket: {$ticket->id}");
                }
            }
        } catch (\Exception $exception) {
            Log::error("STK Callback Exception: " . $exception->getMessage());
        }

        return response()->json([
            'ResponseCode' => '00000000',
            'ResponseDesc' => 'Accept service request successfully',
        ]);
    }

    /**
     * Handle M-Pesa confirmation callback
     */
    public function handleConfirmation(Request $request)
    {
        // Log the confirmation request
        Log::info('M-Pesa Confirmation Callback', ['data' => $request->all()]);

        // {"data":{"TransactionType":"Pay Bill","TransID":"TDS2L38KWI","TransTime":"20250428185618","TransAmount":"1.00","BusinessShortCode":"4083001","BillRefNumber":"4083001","InvoiceNumber":null,"OrgAccountBalance":"1500.00","ThirdPartyTransID":null,"MSISDN":"1f626721b0c2edac4bd8b10899c59f3a460cf3a57e73a9f338db9c4cbec1967e","FirstName":"JOSEPH"}} 


        // Save confirmation details to transactions table
        $transactionData = $request->all();

        Transaction::create([
            'transaction_type' => $transactionData['TransactionType'] ?? null,
            'trans_id' => $transactionData['TransID'] ?? null,
            'trans_time' => $transactionData['TransTime'] ?? null,
            'trans_amount' => $transactionData['TransAmount'] ?? null,
            'business_short_code' => $transactionData['BusinessShortCode'] ?? null,
            'bill_ref_number' => $transactionData['BillRefNumber'] ?? null,
            'invoice_number' => $transactionData['InvoiceNumber'] ?? null,
            'org_account_balance' => $transactionData['OrgAccountBalance'] ?? null,
            'third_party_trans_id' => $transactionData['ThirdPartyTransID'] ?? null,
            'msisdn' => $transactionData['MSISDN'] ?? null,
            'first_name' => $transactionData['FirstName'] ?? null,
        ]);

        // Update ticket payment status based on BillRefNumber (assumed to store the ticket reference)
        if (!empty($transactionData['BillRefNumber'])) {
            $ticket = Ticket::where('reference', $transactionData['BillRefNumber'])->first();

            if ($ticket) {
                $ticket->payment_status = 'completed';
                $ticket->mpesa_receipt_number = $transactionData['TransID'] ?? null;
                $ticket->payment_date = now()->timezone('Africa/Nairobi');
                if (!empty($transactionData['FirstName']) && ($ticket->ticket_type === 'gate' || strtolower($ticket->first_name) === 'gate')) {
                    $ticket->first_name = $transactionData['FirstName'];
                    $ticket->last_name = '';
                }
                $ticket->save();

                // Send sms to the customer
                $phone = $ticket->phone;
                // $message = "Confirmed Ksh. {$ticket->total_amount}, you purchased {$ticket->quantity} early bird tickets on " . now()->timezone('Africa/Nairobi')->format('d/m/Y') . " at " . now()->timezone('Africa/Nairobi')->format('H:i') . ", for {$ticket->event->name} at {$ticket->event->venue} on " . \Carbon\Carbon::parse($ticket->event->date)->timezone('Africa/Nairobi')->format('jS M Y') . ". E-Ticket Ref#: {$ticket->reference}. Download your tickets here: " . url("/get/ticket/{$ticket->reference}");

                $message = "Dear {$ticket->first_name} {$ticket->last_name},Welcome to {$ticket->event->name}  Ticket #: {$ticket->reference}  Tickets: {$ticket->quantity}  Get your tickets here:" . url("/get/ticket/{$ticket->reference}");

                // Send SMS notification
                $this->sendSms($phone, $message);

                // Send email notification
                $this->sendEmail(
                    $ticket->email,
                    "Your PACE TeenzFest Ticket Confirmation",
                    $ticket,
                    $ticket->event
                );

                // Send Discord notification to admin
                // $this->sendDiscordNotification($ticket);

                Log::info("Ticket {$ticket->id} marked as completed via Confirmation callback");
            } else {
                Log::warning('Ticket not found for BillRefNumber: ' . $transactionData['BillRefNumber']);

                // Handle direct paybill payments (no reference or invalid reference)
                // Check if the amount matches the gate fee for Event ID 1
                $event = Event::find(1);
                if ($event) {
                    $gateFee = $event->tickets['gate'] ?? 0;
                    $paidAmount = $transactionData['TransAmount'] ?? 0;

                    // Allow for small differences or exact match? Let's go with exact match or greater for now
                    if ($paidAmount >= $gateFee) {
                        Log::info("Direct Paybill Payment Detected: Amount {$paidAmount} matches gate fee for Event {$event->id}");

                        // Create a new ticket
                        $reference = Ticket::generateReference();

                        // Extract names from transaction
                        $firstName = $transactionData['FirstName'] ?? 'Guest';
                        $lastName = trim(($transactionData['MiddleName'] ?? '') . ' ' . ($transactionData['LastName'] ?? ''));
                        $phone = $transactionData['MSISDN'] ?? '';

                        // Generate a placeholder email if none provided (Paybill doesn't give email)
                        $email = "gate_sale_{$transactionData['TransID']}@pace.co.ke";

                        $ticket = Ticket::create([
                            'event_id' => $event->id,
                            'reference' => $reference,
                            'ticket_type' => 'gate',
                            'first_name' => $firstName,
                            'last_name' => $lastName,
                            'phone' => $phone,
                            'email' => $email,
                            'school' => 'Direct gate payment',
                            'quantity' => 1, // Assuming 1 ticket for the amount
                            'amount' => $paidAmount,
                            'total_amount' => $paidAmount,
                            'payment_status' => 'completed',
                            'mpesa_receipt_number' => $transactionData['TransID'],
                            'payment_date' => now()->timezone('Africa/Nairobi'),
                        ]);

                        // Send notifications
                        $message = "Dear {$firstName} {$lastName},Welcome to {$event->name}  \nTicket #: {$reference}  \nTickets: 1  \nGet your tickets here:" . url("/get/ticket/{$reference}");

                        // Send SMS
                        // Send SMS
                        if ($phone) {
                            $this->sendSms($transactionData['BillRefNumber'], $message);
                        }

                        // Send Discord notification
                        // $this->sendDiscordNotification($ticket);

                        Log::info("Created new direct ticket {$ticket->id} for transaction {$transactionData['TransID']}");
                    }
                }
            }
        }

        // Additional business logic can go here (e.g., updating ticket status)

        // Send notification to admin


        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Confirmation received successfully']);
    }

    /**
     * Handle M-Pesa validation callback
     */
    public function handleValidation(Request $request)
    {
        // Log the validation request
        Log::info('M-Pesa Validation Callback', ['data' => $request->all()]);

        // You can perform validation logic here if needed
        // For example, check if the account number exists or if the customer has sufficient balance

        // Return a success response to accept the transaction
        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Validation successful']);
    }

    /**
     * Send Discord notification to admin when ticket is purchased
     */
    // private function sendDiscordNotification($ticket)
    // {
    //     try {
    //         $webhookUrl = 'https://discord.com/api/webhooks/1389718517795917885/2In2Usw-1hxCGlrPvnbe4rx5pSx1k319AuB4Zn4suRppDYBjAhb8v9zt0z9Rgdu2FFBd';

    //         $event = $ticket->event;
    //         $totalAmount = number_format($ticket->total_amount, 2);
    //         $eventDate = \Carbon\Carbon::parse($event->date)->format('jS M Y');
    //         $purchaseTime = $ticket->payment_date ? $ticket->payment_date->format('d/m/Y H:i') : now()->format('d/m/Y H:i');

    //         $embed = [
    //             'title' => '🎫 New Ticket Purchase!',
    //             'description' => 'A new ticket has been purchased and payment confirmed.',
    //             'color' => 0x00ff00, // Green color
    //             'fields' => [
    //                 [
    //                     'name' => '👤 Customer Details',
    //                     'value' => "**Name:** {$ticket->first_name} {$ticket->last_name}\n**Phone:** {$ticket->phone}\n**Email:** {$ticket->email}\n**School:** {$ticket->school}",
    //                     'inline' => true
    //                 ],
    //                 [
    //                     'name' => '🎟️ Ticket Details',
    //                     'value' => "**Event:** {$event->name}\n**Venue:** {$event->venue}\n**Date:** {$eventDate}\n**Quantity:** {$ticket->quantity}",
    //                     'inline' => true
    //                 ],
    //                 [
    //                     'name' => '💰 Payment Details',
    //                     'value' => "**Amount:** KSH {$totalAmount}\n**Reference:** {$ticket->reference}\n**Receipt:** {$ticket->mpesa_receipt_number}\n**Time:** {$purchaseTime}",
    //                     'inline' => false
    //                 ]
    //             ],
    //             'footer' => [
    //                 'text' => 'PACE TeenzFest 2025 - Admin Notification'
    //             ],
    //             'timestamp' => now()->toISOString()
    //         ];

    //         $payload = [
    //             'username' => 'Ticket Bot',
    //             'avatar_url' => 'https://cdn.discordapp.com/attachments/123456789/ticket-icon.png',
    //             'embeds' => [$embed]
    //         ];

    //         $response = Http::post($webhookUrl, $payload);

    //         if ($response->successful()) {
    //             Log::info('Discord notification sent successfully', [
    //                 'ticket_id' => $ticket->id,
    //                 'reference' => $ticket->reference
    //             ]);
    //         } else {
    //             Log::error('Discord notification failed', [
    //                 'ticket_id' => $ticket->id,
    //                 'reference' => $ticket->reference,
    //                 'status' => $response->status(),
    //                 'response' => $response->body()
    //             ]);
    //         }
    //     } catch (\Exception $e) {
    //         Log::error('Discord notification exception', [
    //             'ticket_id' => $ticket->id,
    //             'reference' => $ticket->reference,
    //             'error' => $e->getMessage()
    //         ]);
    //     }
    // }

    /**
     * Send SMS to a phone number (placeholder implementation).
     */

    public function sendSms($recipientPhone, $smsMessage)
    {
        // $recipientPhone = $request->input('recipientPhone');
        // $smsMessage = $request->input('smsMessage');

        try {
            // Handle input from both HTTP request and direct method call
            if ($recipientPhone instanceof Request) {
                $request = $recipientPhone;
                $recipientPhone = $request->input('recipientPhone');
                $smsMessage = $request->input('smsMessage');
            }

            if (empty($recipientPhone) || empty($smsMessage)) {
                throw new \InvalidArgumentException('Recipient phone and message are required');
            }

            $client = new Client();

            $headers = [
                'Content-Type' => 'application/json',
                'ApiKey' => '2ZqSn5LGPfNyTpvRICWiG3vWnJim6zTR6oXimzg29Isn3niqSrXO8fKrOv4FZDSa',
            ];

            $body = [
                'from' => 'PACESETTER',
                'recipients' => [$recipientPhone],
                'message' => $smsMessage,
            ];

            $response = $client->post('https://api.wasiliana.com/api/v1/send/sms', [
                'headers' => $headers,
                'json' => $body
            ]);

            $responseData = json_decode($response->getBody(), true);
            Log::info('SMS sent successfully', [
                'recipient' => $recipientPhone,
                'response' => $responseData
            ]);

            return response()->json($responseData);
        } catch (\Exception $e) {
            Log::error('SMS sending failed', [
                'recipient' => $recipientPhone,
                'error' => $e->getMessage()
            ]);

            if ($recipientPhone instanceof Request) {
                return response()->json(['error' => 'Failed to send SMS'], 500);
            }
            throw $e;
        }
    }

    private function sendEmail($to, $subject, $ticket, $event)
    {
        try {
            $emailHtml = view('emails.ticket', [
                'ticket' => $ticket,
                'event' => $event,
                'downloadUrl' => url("/get/ticket/{$ticket->reference}")
            ])->render();

            // Check email format + DNS
            if (!filter_var($to, FILTER_VALIDATE_EMAIL)) {
                return response()->json(['error' => 'Invalid email format'], 422);
            }

            if (!checkdnsrr(substr(strrchr($to, "@"), 1), "MX")) {
                return response()->json(['error' => 'Email domain cannot receive mail'], 422);
            }

            $client = new Client();

            $response = $client->post('https://pacesetter.co.ke/api/email-sender.php', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ],
                'json' => [
                    'to' => $to,
                    'subject' => $subject,
                    'html' => $emailHtml
                ]
            ]);

            $responseData = json_decode($response->getBody(), true);
            Log::info('Email sent successfully', [
                'recipient' => $to,
                'response' => $responseData
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Email sending failed', [
                'recipient' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Display a simple ticket download confirmation page.
     */
    public function showTicketDownload(string $ticket)
    {
        // Try to find by ID first, then by reference
        $ticketModel = Ticket::where('id', $ticket)
            ->orWhere('reference', $ticket)
            ->first();

        // If ticket not found, show custom not-found page
        if (!$ticketModel) {
            return view('ticket.not-found');
        }

        $event = $ticketModel->event; // may be null

        return view('ticket.download', [
            'ticket' => $ticketModel,
            'event' => $event,
        ]);
    }

    public function showComplimentaryTicketDownload(string $ticket)
    {

        // Try to find by ID first, then by reference
        $ticketModel = ComplimentaryTicket::where('reference', $ticket)
            ->orWhere('complimentary_code', $ticket)
            ->first();

        // If ticket not found, show custom not-found page
        if (!$ticketModel) {
            return view('ticket.not-found');
        }

        $event = $ticketModel->event; // may be null

        return view('ticket.complimentary-download', [
            'ticket' => $ticketModel,
            'event' => $event,
        ]);

    }

    public function showTicketDownloadv2(string $ticket)
    {
        // Try to find by ID first, then by reference
        $ticketModel = Ticket::where('id', $ticket)
            ->orWhere('reference', $ticket)
            ->first();

        // If ticket not found, show custom not-found page
        if (!$ticketModel) {
            return view('ticket.not-found');
        }

        $event = $ticketModel->event; // may be null

        return view('ticket.download-v2', [
            'ticket' => $ticketModel,
            'event' => $event,
        ]);
    }

    public function ticketStatus($reference)
    {
        $ticket = Ticket::where('reference', $reference)->first();
        if (!$ticket) {
            return response()->json([
                'exists' => false,
                'payment_status' => 'not_found'
            ], 404);
        }
        $txn = TxnLog::where('ticket_reference_number', $reference)->orderByDesc('id')->first();
        return response()->json([
            'exists' => true,
            'payment_status' => $ticket->payment_status,
            'mpesa_receipt_number' => $ticket->mpesa_receipt_number,
            'checkout_request_id' => $txn->checkout_request_id ?? null,
            'stk_status' => $txn->stk_status ?? null,
        ]);
    }

    /**
     * Clean up old failed tickets (older than 24 hours)
     * This can be called periodically or via a scheduled task
     */
    public function cleanupFailedTickets()
    {
        $deletedCount = Ticket::where('payment_status', 'failed')
            ->where('created_at', '<', now()->subHours(24))
            ->delete();

        Log::info('Cleaned up old failed tickets', ['deleted_count' => $deletedCount]);

        return response()->json([
            'status' => 'success',
            'message' => "Cleaned up {$deletedCount} old failed tickets"
        ]);
    }


    /**
     * Display the login view.
     */
    public function create(): View
    {
        return view('auth.login');
    }

    /**
     * Handle an incoming authentication request.
     */
    public function storre(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();

        $request->session()->regenerate();

        return redirect()->intended(route('dashboard', absolute: false));
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }


    /**
     * Detect telecom provider from phone number
     */
    private function telco($phone)
    {
        $phone = trim($phone);

        // Safaricom prefixes
        $safaricom = ['25470', '25471', '25472', '25474', '25479', '25410', '25411', '25412'];
        // Airtel prefixes
        $airtel = ['25473', '25475', '25478', '25401', '25402'];
        // Telkom prefixes
        $telkom = ['25476', '25477'];

        // Check first 5 digits for telecom
        $prefix5 = substr($phone, 0, 5);
        if (in_array($prefix5, $safaricom)) {
            return 'Safaricom';
        } elseif (in_array($prefix5, $airtel)) {
            return 'Airtel';
        } elseif (in_array($prefix5, $telkom)) {
            return 'Telkom';
        }

        return 'Unknown';
    }
}
